/** @file   fpstimer.h
 * @brief   Declaration of FpsTimer - class.
 * @version $Revision: 1.2 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_ENG2D_FPSTIMER_H
#define H_ENG2D_FPSTIMER_H

#include "eng2d_dll.h"

namespace eng2d {


/** @class  FpsTimer
 * @brief   Helps you to control the speed of your application.
 * @author  Tomi Lamminsaari
 *
 * This class provides simple timing functionality that helps you to maintain
 * constant speed in your application on slower computers. This class has
 * both static interface as well as methods that needs instances of
 * FpsTimer.
 *
 * First you should call the method <code>static int install(int aBps)</code>
 * to install the timer interruption handler and to define the desired speed
 * of your application. Next you can create an instance of FpsTimer by
 * calling the <code>static FpsTimer* getInstance()</code>.
 *
 * In the mainloop of your application you can use this istance to ask how
 * many updates we should make before we should redraw the screen. After each
 * update round you've done, you must inform the FpsTimer by calling the
 * <code>void updateDone()</code> - method. Here's an example.
 *
 * @code
 *  FpsTimer* gameTimer = FpsTimer::getInstance();
 *  while ( !key[KEY_ESC] ) {
 *    bool needsRedraw = false;
 *    if ( gameTimer->updatesToGo() > 0 ) {
 *      do {
 *        // We update our game structures
 *        update();
 *        // We tell our timer that we've done an update.
 *        gameTimer->updateDone();
 *        // And we set the flag that indicates that we should
 *        // redraw the screen.
 *        needsRedraw = true;
 *      while ( gameTimer->updatesToGo() > 0 );
 *    }
 *    if ( needsRedraw == true ) {
 *      // We redraw the game
 *      redraw();
 *      needsRedraw = false;
 *    }
 *  }
 * @endcode
 */
class DLLIMPORT FpsTimer
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================
  
  
  /** An errorcode returned when there was no error. */
  static const int KNone = 0;
  /** An errorcode returned when invalid bps-value was given. */
  static const int KBpsOutOfRange = 1;
  /** An errorcode returned when Allegro - library was unable to install the
   * interrup handler.
   */
  static const int KAllegroError = 2;
  
  
  /** Installs the timerhandler that is updated with given number of times
   * per second.
   * @param     aBps              Beats per Second. Number of updates per second
   *                              Valid values are 1 ... 1000
   * @return    FpsTimer::KNone returned when there were no errors. Other
   *            possible errorcodes are FpsTimer::BpsOutOfRange and
   *            FpsTimer::KAllegroError
   */
  static int install( int aBps );
  
  /** Removes the timerhandler interrup.
   */
  static void remove();
  
  /** Returns a pointer to an instance of FpsTimer that you can use. You must
   * not delete this object. When you don't need it anymore, you must
   * call the @c releaseInstance(...) - method.
   * @return    Pointer to FpsTimer
   */
  static FpsTimer* getInstance();
  
  /** Releases the instance of FpsTimer when you no longer need it.
   * @param     aTimer            Pointer to the instance you got from
   *                              <code>getInstance()</code> - method.
   */
  static void releaseInstance( FpsTimer* aTimer );
  
  
  
  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Destructor
   */
	virtual ~FpsTimer();
	
protected:

	/** Constructor.
   */
	FpsTimer();

	/** Copy constructor.
   * @param     rO                Reference to another FpsTimer
   */
  FpsTimer( const FpsTimer& rO );

	/** Assignment operator
   * @param     rO                Reference to another FpsTimer
   * @return    Reference to us.
   */
  FpsTimer& operator = ( const FpsTimer& rO );

public:


  ///
  /// Methods
  /// =======

  /** Resets the update-counter. After you've called this, the updatesToGo() -
   * method returns 0.
   */
  void reset();

  /** Informs us that one update-round was done. You must call this method
   * in your main update-loop. Otherwise this FpsTimer will not work
   * properly.
   */
  void updateDone();
  

  ///
  /// Getter methods
  /// ==============
  
  /** Returns the number of updates we should do before next redraw.
   * @return    Number of updates we should do before next update.
   */
  int updatesToGo() const;
  
  /** Returns the total number of updates we've done since we installed
   * the timer handler.
   * @return    Total number of updates.
   */
  unsigned int totalUpdatesDone() const;
  
  /** Returns the logical updates count. This count increases each time you
   * call the updateDone() method.
   * @return  Number of logical updates since the start up of the timer.
   */
  unsigned int logicalUpdatesCount() const;


protected:

  /** A flag that indicates if the interrupt handler has been installed. */
  static bool timerInstalled;
  
  /** A member that counts the updates since previous redraw. */
  static volatile int updateCounter;
  
  /** Number of total updates. */
  static volatile unsigned int numOfUpdates;
  
  /** Number of logical updates done so far. */
  static volatile unsigned int logicalUpdatesCountVar;
  
  /** The interrupt handler function itself. */
  static void timerUpdater();
  
  /** Pointer to the instance of FpsTimer once created. */
  static FpsTimer* s_timerInstance;
  
  /** Number of FpsTimer's there still are in use. */
  static int numberOfInstances;
  
  ///
  /// Members
  /// =======



private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif
